<?php declare(strict_types=1);

namespace Plugin\tnt_data_privacy;

use JTL\Customer\CustomerGroup;
use JTL\DB\DbInterface;
use JTL\Language\LanguageHelper;
use JTL\Plugin\Data\Config;
use JTL\Plugin\PluginInterface;
use JTL\Router\Route;
use JTL\Session\Backend;
use JTL\Session\Frontend;
use JTL\Shop;

/**
 *
 */
class Helper
{
    /**
     * @var DbInterface $db
     */
    private DbInterface $db;
    /**
     * @var PluginInterface $extension
     */
    private PluginInterface $extension;
    /**
     * @var Config $config
     */
    private Config $config;

    public const CHECKBOX_CNAME = 'ZustimmungZurDatenuebermittlung';

    /**
     * @param PluginInterface $extension
     * @param DbInterface $db
     */
    public function __construct(PluginInterface $extension, DbInterface $db)
    {
        $this->extension = $extension;
        $this->db        = $db;
        $this->config    = $this->extension->getConfig();
    }

    /**
     * @return int
     */
    public function getCheckBoxID(): int
    {
        $checkBox = $this->db->getSingleObject(
            'SELECT kCheckBox FROM tcheckbox WHERE cName = :cName',
            ['cName' => self::CHECKBOX_CNAME]
        );

        return (int)$checkBox->kCheckBox ?? 0;
    }

    /**
     * @return string
     */
    public function getCheckBoxAdminPath(): string
    {
        $checkBoxID = $this->getCheckBoxID();
        if ($checkBoxID <= 0) {
            return '';
        }

        return Shop::getAdminURL() . '/' . Route::CHECKBOX . '?edit=' . $checkBoxID . '&token='
            . Backend::get('jtl_token');
    }

    /**
     * @return int
     */
    public function insertCheckBox(): int
    {
        $kLink = (int)$this->db->getSingleObject(
            'SELECT kLink FROM tlink WHERE nLinkart = :nLinkart',
            ['nLinkart' => \LINKTYP_DATENSCHUTZ]
        )->kLink ?? 0;

        $cKundengruppe = ';';
        $cgIDs         = CustomerGroup::getGroups();
        foreach ($cgIDs as $cgID) {
            $cKundengruppe .= $cgID->getID() . ';';
        }
        $checkBox                    = new \stdClass();
        $checkBox->kLink             = $kLink;
        $checkBox->kCheckBoxFunktion = 0;
        $checkBox->cName             = self::CHECKBOX_CNAME;
        $checkBox->cKundengruppe     = $cKundengruppe;
        $checkBox->cAnzeigeOrt       = ';' . \CHECKBOX_ORT_BESTELLABSCHLUSS . ';';
        $checkBox->nAktiv            = 1;
        $checkBox->nPflicht          = 0;
        $checkBox->nLogging          = 1;
        $checkBox->nSort             = 1;
        $checkBox->dErstellt         = \date('Y-m-d H:i:s');
        if (intval(str_replace('.', '', \APPLICATION_VERSION)) >= 530) {
            $checkBox->nInternal = 1;
        }

        return $this->db->insertRow(
            'tcheckbox',
            $checkBox
        );
    }

    /**
     * @param int $checkBoxID
     */
    public function insertCheckBoxText(int $checkBoxID): void
    {
        $titles       = [
            'ger' => 'Zustimmung zur Datenübermittlung',
            'eng' => 'Consent to data transmission'
        ];
        $descriptions = [
            'ger' => 'Hiermit bestätige ich, das meine E-Mailadresse und Telefonnummer an '
                . 'den Versanddienstleister zum Zwecke der Information über den '
                . 'Sendungsstatus weitergegeben werden dürfen. Diese Einstellung wird für zukünftige '
                . 'Bestellungen gespeichert und kann jederzeit wiederrufen werden.',
            'eng' => 'I hereby confirm that my email address and telephone number may be '
                . 'transmitted to the shipping service provider for the purpose of '
                . 'providing information on the shipment status. This setting will be saved for future '
                . 'orders and can be revoked at any time.'
        ];

        foreach (LanguageHelper::getAllLanguages() as $language) {
            $languageCode = $language->getCode();
            if (!isset($titles[$languageCode])) {
                continue;
            }
            $checkBoxSprache                = new \stdClass();
            $checkBoxSprache->kCheckBox     = $checkBoxID;
            $checkBoxSprache->kSprache      = $language->getId();
            $checkBoxSprache->cText         = $titles[$languageCode];
            $checkBoxSprache->cBeschreibung = $descriptions[$languageCode];

            $this->db->insertRow(
                'tcheckboxsprache',
                $checkBoxSprache
            );
        }
    }

    /**
     * @return void
     */
    public function removeCheckBox(): void
    {
        $this->db->executeQueryPrepared(
            'DELETE tcheckbox, tcheckboxsprache 
            FROM tcheckbox
            INNER JOIN tcheckboxsprache 
                ON tcheckboxsprache.kCheckBox = tcheckbox.kCheckBox
            WHERE tcheckbox.cName = :cName',
            ['cName' => self::CHECKBOX_CNAME]
        );
    }

    /**
     * @param $args
     * @return void
     */
    public function saveOrderAttribute($args): void
    {
        if (!isset($args['oBestellung'])) {
            return;
        }

        $checkBoxID = $this->getCheckBoxID();

        if ($checkBoxID <= 0) {
            return;
        }

        $this->db->executeQueryPrepared(
            'DELETE FROM tnt_data_privacy WHERE customerID = :customerID',
            ['customerID' => (int)$args['oBestellung']->kKunde]
        );

        $active = $_POST['CheckBox_' . $checkBoxID] ?? 0;
        if ($active !== 0) {
            $active = $_POST['CheckBox_' . $checkBoxID] === 'on' ? 1 : 0;
        }

        $saveSettings             = new \stdClass();
        $saveSettings->customerID = (int)$args['oBestellung']->kKunde;
        $saveSettings->active     = $active;

        $this->db->insertRow(
            'tnt_data_privacy',
            $saveSettings
        );

        if (!isset($_POST['CheckBox_' . $checkBoxID])
            || $_POST['CheckBox_' . $checkBoxID] !== 'on'
        ) {
            return;
        }

        $orderAttribute              = new \stdClass();
        $orderAttribute->kBestellung = (int)$args['oBestellung']->kBestellung;
        $orderAttribute->cName       = $this->extension->getConfig()->getValue('tnt_data_privacy_attribute');
        $orderAttribute->cValue      = 'on';

        $this->db->insertRow(
            'tbestellattribut',
            $orderAttribute
        );
    }

    /**
     * @return void
     */
    public function enableCheckBox(): void
    {
        $this->db->executeQueryPrepared(
            'UPDATE tcheckbox SET nAktiv = 1 WHERE cName = :cName',
            ['cName' => self::CHECKBOX_CNAME]
        );
    }

    /**
     * @return void
     */
    public function disableCheckBox(): void
    {
        $this->db->executeQueryPrepared(
            'UPDATE tcheckbox SET nAktiv = 0 WHERE cName = :cName',
            ['cName' => self::CHECKBOX_CNAME]
        );
    }

    /**
     * @return void
     */
    public function toggleCheckBox(): void
    {
        $checkBoxID = $this->getCheckBoxID();
        if ($checkBoxID <= 0) {
            return;
        }

        $active = (int)$this->db->getSingleObject(
            'SELECT active FROM tnt_data_privacy WHERE customerID = :customerID',
            ['customerID' => Frontend::getCustomer()->getID() ?? 0]
        )->active ?? 0;

        if ($active === 1) {
            Shop::Smarty()->assign(
                'cPost_arr',
                ['CheckBox_' . $checkBoxID => 'on']
            );
        }
    }
}
