<?php declare(strict_types=1);

namespace Plugin\tnt_cartzilla;

use Exception;
use JTL\Events\Dispatcher;
use JTL\Helpers\Form;
use JTL\Helpers\Request;
use JTL\Language\LanguageHelper;
use JTL\Catalog\Product\Artikel;
use JTL\Catalog\Category\Kategorie;
use JTL\Plugin\Bootstrapper;
use JTL\Shop;
use JTL\Session\Frontend;
use JTL\Smarty\JTLSmarty;
use SimpleXMLElement;
use stdClass;
use ZipArchive;

class Bootstrap extends Bootstrapper
{

    private $license;

    public function boot(Dispatcher $dispatcher)
    {
        parent::boot($dispatcher);
        $dispatcher->listen('shop.hook.' . \HOOK_IO_HANDLE_REQUEST, [$this, 'ioRequest']);
        //HOOK_FILTER_INC_GIBARTIKELKEYS
        $dispatcher->listen('shop.hook.' . \HOOK_PRODUCTFILTER_INIT_STATES, [$this, 'filterEnde']);
        $dispatcher->listen('shop.hook.' . \HOOK_LOCALIZED_PRICE_STRING, [$this, 'smPrice']);
        //$dispatcher->listen('shop.hook.' . \HOOK_LETZTERINCLUDE_INC, [$this, "getFilterArgs"]);
        $dispatcher->listen('shop.hook.' . \HOOK_BEWERTUNG_INC_SPEICHERBEWERTUNG, [$this, 'saveBewertung']);
        $dispatcher->listen('shop.hook.' . \HOOK_KUNDE_CLASS_LOADFROMDB, [$this, 'avatar']);
        $dispatcher->listen('shop.hook.' . \HOOK_JTL_PAGE_KUNDENACCOUNTLOESCHEN, [$this, 'avatar']);
        $dispatcher->listen('shop.hook.' . \HOOK_LETZTERINCLUDE_INC, [$this, 'getConfig']);
    }

    private function getLicense(): bool
    {
        $license = $this->getPlugin()->getLicense()->getExsLicense();
        if ($license !== null) {
            $this->license = $license->getExsID() ?? '';
            if ($this->license !== '') {
                return true;
            }
        }
        Shop::Container()->getAlertService()->addWarning(
            'A license for this plugin cant be found',
            'LicenseError'
        );
        return false;
    }

    public function getConfig(array $args)
    {
        if (isset($args[0])) {
            if ($args[0] == \HOOK_LETZTERINCLUDE_INC) {
                $db     = Shop::Container()->getDB();
                $smarty = Shop::Smarty();
                $config = $db->selectSingleRow(
                    'cartzilla_config',
                    array('tkey', 'tvalue'),
                    array('config', 'template')
                );
                $smarty->assign('cz_config', $this->getConfigValue($config));
            }
        }
    }

    private function getConfigValue($config, $key = '')
    {
        $result = json_decode($config->textra ?? '{}');
        if ($key !== '') {
            if (isset($result->{$key})) {
                $result = $result->{$key};
            } else {
                $result = null;
            }
        }
        return $result;
    }

    private function uploadAvatarImage(array $tmpFile): array
    {
        $Kunde = Frontend::getCustomer();
        if (isset($Kunde->kKunde)) {
            $file    = [
                'type' => $tmpFile['type'],
                'tmp_name' => $tmpFile['tmp_name'],
                'error' => $tmpFile['error'],
                'name' => $tmpFile['name']
            ];
            $imgType = \array_search($file['type'], [
                \IMAGETYPE_JPEG => \image_type_to_mime_type(\IMAGETYPE_JPEG),
                \IMAGETYPE_PNG => \image_type_to_mime_type(\IMAGETYPE_PNG),
                \IMAGETYPE_BMP => \image_type_to_mime_type(\IMAGETYPE_BMP),
                \IMAGETYPE_GIF => \image_type_to_mime_type(\IMAGETYPE_GIF),
            ], true);
            if ($imgType === false) {
                return array();
            }

            $imagePath = 'bilder/avatare/';
            $uploadDir = \PFAD_ROOT . $imagePath;
            $imageName = $Kunde->kKunde . \image_type_to_extension($imgType);
            if (\is_dir($uploadDir) || (\mkdir($uploadDir, 0755) && \is_dir($uploadDir))) {
                if (\move_uploaded_file($file['tmp_name'], \PFAD_ROOT . $imagePath . $imageName)) {
                    return array($Kunde->kKunde, '/' . $imagePath . $imageName);
                }
            }
        }
        return array();
    }

    public function avatar(array $args): void
    {
        $db = Shop::Container()->getDB();
        if (isset($args[0])) {
            if ($args[0] == \HOOK_KUNDE_CLASS_LOADFROMDB) {
                if (isset($_FILES['avatar'])) {
                    if (isset($_FILES['avatar']) && !empty($_FILES['avatar']['tmp_name'])) {
                        $img = $this->uploadAvatarImage($_FILES['avatar']);
                        if (count($img) > 0) {
                            $obj         = new stdClass();
                            $obj->tkey   = 'avatar';
                            $obj->tvalue = $img[0];
                            $obj->textra = $img[1];
                            $db->deleteRow('cartzilla_config', ['tkey', 'tvalue'], ['avatar', strval($img[0])]);
                            $db->insertRow('cartzilla_config', $obj);
                        }
                    }
                }
            } elseif ($args[0] == \HOOK_JTL_PAGE_KUNDENACCOUNTLOESCHEN) {
                $Kunde = Frontend::getCustomer();
                if (isset($Kunde->kKunde)) {
                    $imagePath = \PFAD_ROOT . 'bilder/avatare/';
                    $db->deleteRow('cartzilla_config', ['tkey', 'tvalue'], ['avatar', strval($Kunde->kKunde)]);
                    $files = glob($imagePath . $Kunde->kKunde . '.*');
                    foreach ($files as $file) {
                        unlink($file);
                    }
                }
            }
        }
    }

    public function saveBewertung(array $args): void
    {
        if (isset($args['rating'])) {
            if (isset($_POST['pros'])) {
                if ($_POST['pros'] != '') {
                    $_POST['pros']            = strip_tags($_POST['pros']);
                    $args['rating']->content .= '{Pros}' . $_POST['pros'] . '{/Pros}';
                }
            }
            if (isset($_POST['cons'])) {
                if ($_POST['cons'] != '') {
                    $_POST['cons']            = strip_tags($_POST['cons']);
                    $args['rating']->content .= '{Cons}' . $_POST['cons'] . '{/Cons}';
                }
            }
        }
    }

    public function getFilterArgs(array $args): void
    {
        $filter       = Shop::getProductFilter();
        $manufacturer = $filter->getFilterByClassName('JTL\Filter\Items\Manufacturer');
        //$manufacturer->getUrlParam();
    }

    public function smPrice(array $args): void
    {
        $args['localized'] = preg_replace(
            '/([0-9.,]+)([0-9]{2})/',
            "$1<small class='d-inline-block decimals'>$2</small>",
            $args['localized']
        );
    }

    /* Zeige in den Oberkategorien auch die Artikel aus den Unterkategorien an */
    public function filterEnde(array $args): void
    {
        $baseCat = $args['productFilter']->getCategory();
        $baseCat->setIncludeSubCategories(true);
    }

    public function ioRequest(array $args): void
    {
        $io = $args['io'];
        try {
            $io->register('tnt_suggestions', [$this, 'suggestions']);
        } catch (Exception $e) {
            Shop::Container()->getLogService()->error('Could not register io requests: ' . $e);
        }
    }

    public function suggestions($keyword): array
    {
        $results = [];
        if (\mb_strlen($keyword) < 2) {
            return $results;
        }
        $smarty     = Shop::Smarty();
        $language   = Shop::getLanguageID();
        $maxResults = ($cnt = Shop::getSettingValue(\CONF_ARTIKELUEBERSICHT, 'suche_ajax_anzahl')) > 0
            ? $cnt
            : 10;
        /* MODIFIED 24/08/2022 */
        $db              = $this->getDB();
        $lang_is_default = LanguageHelper::isDefaultLanguageActive(true, $language);
        if ($lang_is_default) {
            $srch_results = $db->getObjects(
                'SELECT tartikel.kArtikel, tartikel.cName
					FROM tartikel
					WHERE tartikel.cName LIKE CONCAT(\'%\', TRIM(:keyword), \'%\')
						OR tartikel.cSuchbegriffe LIKE CONCAT(\'%\', TRIM(:keyword), \'%\')
					LIMIT :maxres',
                [
                    'keyword' => $keyword,
                    'maxres' => $maxResults
                ]
            );
            $searchCat    = $db->getObjects(
                'SELECT tkategorie.kKategorie
                    FROM tkategorie
                    WHERE tkategorie.cName LIKE CONCAT(\'%\', TRIM(:keyword), \'%\')
                    LIMIT :maxres',
                [
                    'keyword' => $keyword,
                    'maxres' => $maxResults
                ]
            );
        } else {
            $srch_results = $db->getObjects(
                'SELECT tartikelsprache.cName, tartikelsprache.kArtikel
                    FROM tartikelsprache
                    WHERE tartikelsprache.cName LIKE CONCAT(\'%\', TRIM(:keyword), \'%\')
                      AND kSprache = :lang
                    LIMIT :maxres',
                [
                    'keyword' => $keyword,
                    'maxres' => $maxResults,
                    'lang' => $language
                ]
            );
            $searchCat    = $db->getObjects(
                'SELECT tkategoriesprache.kKategorie
                    FROM tkategoriesprache
                    WHERE tkategoriesprache.cName LIKE CONCAT(\'%\', TRIM(:keyword), \'%\')
                      AND tkategoriesprache.kSprache = :lang
                    LIMIT :maxres',
                [
                    'keyword' => $keyword,
                    'maxres' => $maxResults,
                    'lang' => $language
                ]
            );
        }
        $smarty->assign('shopURL', Shop::getURL());
        $suggestions = '';
        $limit       = $maxResults;
        if ($limit <= 0) {
            $limit = 1;
        }
        foreach ($srch_results as $result) {
            if ($limit <= 0) {
                break;
            }
            $product = new Artikel();
            $options = Artikel::getDefaultOptions();
            $product->fuelleArtikel(intval($result->kArtikel), $options, 0, 0, false);
            $suggestions .= $smarty->assign('result', $product)->fetch('snippets/suggestion.tpl');
            $limit--;
        }
        foreach ($searchCat as $result) {
            if ($limit <= 0) {
                break;
            }
            $category     = new Kategorie(intval($result->kKategorie));
            $suggestions .= $smarty->assign('result', $category)->assign('type', 'category')
                ->fetch('snippets/suggestion.tpl');
            $limit--;
        }
        if ($suggestions === '') {
            $langHelper  = LanguageHelper::getInstance();
            $suggestions = $langHelper->getTranslation('no_search_results', 'custom');
        }
        $tmp             = new stdClass();
        $tmp->suggestion = $smarty->assign('tnt_search_suggestions', $suggestions)->fetch(
            'snippets/search_suggestions_dropdown.tpl'
        );
        $results[]       = $tmp;
        return $results;
    }

    /**
     * @param string $url
     * @param string $zip
     * @return bool|mixed
     */
    private function curlTNT(string $url, string $zip = ''): mixed
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120);
        if ($zip !== '') {
            $fh = fopen($zip, 'w+');
            curl_setopt($ch, CURLOPT_FILE, $fh);

            curl_setopt($ch, CURLOPT_HEADER, 0);
            curl_setopt(
                $ch,
                CURLOPT_USERAGENT,
                'Mozilla/5.0 (Windows NT 6.2; WOW64; rv:17.0) Gecko/20100101 Firefox/17.0'
            );
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);

            $rawZipFile = curl_exec($ch);
            if (curl_errno($ch)) {
                echo 'error:' . curl_error($ch);
            }
            fwrite($fh, $rawZipFile);
            curl_close($ch);
            fclose($fh);
            return filesize($zip) > 0;
        } else {
            //curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            $result = json_decode(curl_exec($ch));

            if (curl_errno($ch)) {
                echo 'error:' . curl_error($ch);
            }
            curl_close($ch);
            return $result;
        }
    }

    private function rrmdir($dir)
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != '.' && $object != '..') {
                    if (is_dir($dir . DIRECTORY_SEPARATOR . $object) && !is_link($dir . '/' . $object)) {
                        $this->rrmdir($dir . DIRECTORY_SEPARATOR . $object);
                    } else {
                        unlink($dir . DIRECTORY_SEPARATOR . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }

    /**
     * @throws Exception
     */
    private function upgrade(): bool
    {
        if (!$this->getLicense()) {
            return false;
        }

        $installed       = false;
        $templateDir     = dirname(__FILE__, 3) . '/templates/cartzilla/';
        $zipFileLocation = dirname(__FILE__, 3) . '/templates/cz_update/';

        if (!file_exists($zipFileLocation)) {
            mkdir($zipFileLocation, 0755, true);
        } else {
            $this->rrmdir($zipFileLocation . 'cartzilla');
        }
        if (!$this->curlTNT(
            'https://api.timniko.com/get/zip/id/' . $this->license,
            $zipFileLocation . 'cartzilla.zip'
        )) {
            return false;
        }
        $archive = new ZipArchive();
        if (($archive->open($zipFileLocation . 'cartzilla.zip')) === true) {
            if ($archive->extractTo($zipFileLocation)) {
                $foldersInZip = glob($zipFileLocation . '*', GLOB_ONLYDIR);
                if (count($foldersInZip) > 1) {
                    $this->rrmdir($templateDir);
                    if (rename($zipFileLocation, $templateDir)) {
                        unlink($zipFileLocation . 'cartzilla.zip');
                        rmdir($zipFileLocation);
                        $installed = true;
                    }
                    if ($installed) {
                        $db         = Shop::Container()->getDB();
                        $newVersion = '';
                        try {
                            $xmlFile    = new SimpleXMLElement(dirname(__FILE__, 3) .
                                '/templates/cartzilla/template.xml', 0, true);
                            $newVersion = (string)$xmlFile->Version ?? '';
                        } catch (Exception $e) {
                            var_dump($e);
                            exit();
                        }
                        $upd          = new stdClass();
                        $upd->version = $newVersion;
                        $db->updateRow('ttemplate', 'cTemplate', 'cartzilla', $upd);
                        $tsprachwerte = $templateDir . 'tsprachwerte.sql';
                        if (file_exists($tsprachwerte)) {
                            $dbObj = $db->selectSingleRow('tsprachsektion', 'cName', 'custom');
                            $engID = null;
                            $gerID = null;
                            if (isset($dbObj->kSprachsektion)) {
                                $kSprachsektion = intval($dbObj->kSprachsektion);
                                $dbObj          = $db->selectSingleRow('tsprache', 'cISO', 'eng');
                                if (isset($dbObj->kSprache)) {
                                    $engID = intval($dbObj->kSprache);
                                }
                                $dbObj = $db->selectSingleRow('tsprache', 'cISO', 'ger');
                                if (isset($dbObj->kSprache)) {
                                    $gerID = intval($dbObj->kSprache);
                                }
                                if (!is_null($engID) || !is_null($gerID)) {
                                    $sql = file_get_contents($tsprachwerte);
                                    if (!is_null($engID)) {
                                        $sql = preg_replace(
                                            "/\(2, 18, '/",
                                            '(' . $engID . ', ' . $kSprachsektion . ", '",
                                            $sql
                                        );
                                    }
                                    if (!is_null($gerID)) {
                                        $sql = preg_replace(
                                            "/\(1, 18, '/",
                                            '(' . $gerID . ', ' . $kSprachsektion . ", '",
                                            $sql
                                        );
                                    }
                                    $db->executeQuery($sql);
                                }
                            }
                            unlink($tsprachwerte);
                        }
                    }
                }
                $archive->close();
                unlink($templateDir . 'cartzilla.zip');
            }
        }
        return $installed;
    }

    private function getHelper($db, $name): object
    {
        $helper      = new stdClass();
        $tpl_path    = str_replace('tnt_cartzilla', 'tnt_cartzilla_' . $name, __DIR__) . '/extra.tpl';
        $helper_path = str_replace('tnt_cartzilla', 'tnt_cartzilla_' . $name, __DIR__) . '/helper.php';
        if (file_exists($helper_path) && file_exists($tpl_path)) {
            $dbObj = $db->selectSingleRow('tplugin', 'cPluginID', 'tnt_cartzilla_' . $name);
            if (isset($dbObj->nStatus) && intval($dbObj->nStatus == 2)) {
                require_once($helper_path);
                $classname = 'cz_' . $name . '_Helper';
                if (class_exists($classname)) {
                    $helper = new $classname($this->getDB(), $tpl_path);
                }
            }
        }
        return $helper;
    }

    public function renderAdminMenuTab(string $tabName, int $menuID, JTLSmarty $smarty): string
    {
        $plugin = $this->getPlugin();
        $smarty->assign('plugin_pfad', Shop::getURL() . '/plugins/tnt_cartzilla/');
        $smarty->assign('admin_pfad', Shop::getURL() . '/admin/');
        $smarty->assign('tpl_news', $smarty->fetch(__DIR__ . '/news.tpl'));

        $db             = $this->getDB();
        $css_additional = '
                <link rel="stylesheet" media="screen" href="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/simplebar/dist/simplebar.min.css"/>
                <link rel="stylesheet" media="screen" href="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/tiny-slider/dist/tiny-slider.css"/>
            ';
        $js_additional  = '<script src="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/bootstrap/dist/js/bootstrap.bundle.min.js"></script>'
            . '<script src="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/simplebar/dist/simplebar.min.js"></script>'
            . '<script src="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/tiny-slider/dist/min/tiny-slider.js"></script>'
            . '<script src="'
            . Shop::getURL()
            . '/plugins/tnt_cartzilla/adminmenu/vendor/smooth-scroll/dist/smooth-scroll.polyfills.min.js"></script>';
        $badge_check    = '<span class="badge bg-success"><i class="ci-check" style="font-size: .5rem;"></i></span>';
        $badge_nocheck  = '<span class="badge bg-primary"><i class="ci-close" style="font-size: .5rem;"></i></span>';

        $helperSilder = $this->getHelper($db, 'slider');
        if (method_exists($helperSilder, 'get_tpl')) {
            $smarty->assign('tpl_slider', $helperSilder->get_tpl());
            $smarty->assign('plg_slider', $badge_check);
            $css_additional .= $helperSilder->get_css();
            $js_additional  .= $helperSilder->get_js();
        } else {
            $smarty->assign('tpl_slider', $smarty->fetch(__DIR__ . '/extra.tpl'));
            $smarty->assign('plg_slider', $badge_nocheck);
        }

        $helperAdBanner = $this->getHelper($db, 'adbanner');
        if (method_exists($helperAdBanner, 'get_tpl')) {
            $smarty->assign('tpl_adbanner', $helperAdBanner->get_tpl());
            $smarty->assign('plg_adbanner', $badge_check);
            $css_additional .= $helperAdBanner->get_css();
            $js_additional  .= $helperAdBanner->get_js();
        } else {
            $smarty->assign('tpl_adbanner', $smarty->fetch(__DIR__ . '/extra.tpl'));
            $smarty->assign('plg_adbanner', $badge_nocheck);
        }

        $helperFarbfilter = $this->getHelper($db, 'farbfilter');
        if (method_exists($helperFarbfilter, 'get_tpl')) {
            $smarty->assign('tpl_farbfilter', $helperFarbfilter->get_tpl());
            $smarty->assign('plg_farbfilter', $badge_check);
            $css_additional .= $helperFarbfilter->get_css();
            $js_additional  .= $helperFarbfilter->get_js();
        } else {
            $smarty->assign('tpl_farbfilter', $smarty->fetch(__DIR__ . '/extra.tpl'));
            $smarty->assign('plg_farbfilter', $badge_nocheck);
        }

        $helperBestellungen = $this->getHelper($db, 'bestellungen');
        if (method_exists($helperBestellungen, 'get_tpl')) {
            $smarty->assign('tpl_bestellungen', $helperBestellungen->get_tpl());
            $smarty->assign('plg_bestellungen', $badge_check);
            $css_additional .= $helperBestellungen->get_css();
            $js_additional  .= $helperBestellungen->get_js();
        } else {
            $smarty->assign('tpl_bestellungen', $smarty->fetch(__DIR__ . '/extra.tpl'));
            $smarty->assign('plg_bestellungen', $badge_nocheck);
        }

        $smarty->assign('css_additional', $css_additional);
        $smarty->assign('js_additional', $js_additional);

        $templateDir = dirname(__FILE__, 3) . '/templates/cartzilla/';
        if (file_exists($templateDir)) {
            $smarty->assign('tVorhanden', 'Y');
        } else {
            $smarty->assign('tVorhanden', 'N');
        }
        if (Form::validateToken()) {
            if ((Request::postVar('update_install')) !== null) {
                $update            = $this->upgrade();
                $response          = new stdClass();
                $response->type    = $update ? 'success' : 'danger';
                $response->message = $update ? 'Update successful' : 'Update failed';
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode($response);
                die();
            } elseif (($inhalt = Request::postVar('inhalt')) !== null) {
                $result = '';
                $tpl    = '/pages/contact_form_content.tpl';
                if ($inhalt === '404') {
                    $tpl = '/pages/404.tpl';
                } elseif ($inhalt === 'about') {
                    $tpl = '/pages/about_us.tpl';
                }
                if (file_exists(__DIR__ . $tpl)) {
                    $result = htmlspecialchars(file_get_contents(__DIR__ . $tpl));
                }
                echo $result;
                die();
            } elseif ((Request::postVar('cz_config')) !== null) {
                unset($_POST['cz_config']);
                unset($_POST['jtl_token']);
                $obj         = new stdClass();
                $obj->tkey   = 'config';
                $obj->tvalue = 'template';
                $obj->textra = json_encode($_POST);
                $db->deleteRow('cartzilla_config', ['tkey', 'tvalue'], ['config', 'template']);
                $db->insertRow('cartzilla_config', $obj);
            }
        }
        $config = $db->selectSingleRow('cartzilla_config', ['tkey', 'tvalue'], ['config', 'template']);
        $config = $config->textra ?? '{}';
        $config = addslashes($config);
        $smarty->assign('cz_config', $config)->assign('tpl_config', $smarty->fetch(__DIR__ . '/config.tpl'));
        $template = 'einstellungen.tpl';
        return $smarty->assign(
            'adminURL',
            Shop::getAdminURL() . '/plugin.php?kPlugin=' . $plugin->getID()
        )->fetch($this->getPlugin()->getPaths()->getAdminPath() . '/templates/' . $template);
    }


    public function installed()
    {
        parent::installed();
        if ($this->upgrade()) {
            Shop::Container()->getAlertService()->addSuccess(
                'Your template has been updated.',
                'UpdateSuccess'
            );
        } else {
            Shop::Container()->getAlertService()->addWarning(
                'An error occurred while trying to update your template.',
                'UpdateError'
            );
        }
        Shop::Container()->getCache()->flushTags(
            [CACHING_GROUP_CORE, CACHING_GROUP_TEMPLATE, CACHING_GROUP_OPTION]
        );
    }

    public function updated($oldVersion, $newVersion)
    {
        \error_log('updated from ' . $oldVersion . ' to ' . $newVersion);
    }

    public function uninstalled(bool $deleteData = true)
    {
        parent::uninstalled($deleteData);
    }
}
